/*
 * Decompiled with CFR 0.152.
 */
package net.pterodactylus.util.storage;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.pterodactylus.util.io.Closer;
import net.pterodactylus.util.logging.Logging;
import net.pterodactylus.util.storage.Allocation;
import net.pterodactylus.util.storage.Factory;
import net.pterodactylus.util.storage.Storable;
import net.pterodactylus.util.storage.StorageException;
import net.pterodactylus.util.validation.Validation;

public class Storage<T extends Storable>
implements Closeable {
    private static final Logger logger = Logging.getLogger(Storage.class);
    private final int blockSize;
    private final Factory<T> factory;
    private RandomAccessFile indexFile;
    private RandomAccessFile dataFile;
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final List<Allocation> directoryEntries = new ArrayList<Allocation>();
    private final Map<Long, Integer> idDirectoryIndexes = new HashMap<Long, Integer>();
    private final BitSet emptyDirectoryEntries = new BitSet();
    private final BitSet allocations = new BitSet();
    private boolean opened;
    private final File directory;
    private final String name;

    public Storage(Factory<T> factory, File directory, String name) {
        this(512, factory, directory, name);
    }

    public Storage(int blockSize, Factory<T> factory, File directory, String name) {
        Validation.begin().isNotNull("Factory", factory).isNotNull("Directory", directory).isNotNull("Name", name).check().isGreater("Block Size", blockSize, 0L).check();
        this.blockSize = blockSize;
        this.factory = factory;
        this.directory = directory;
        this.name = name;
    }

    public void open() throws StorageException {
        logger.log(Level.INFO, "[%s] Opening Storage\u2026", this.name);
        this.lock.writeLock().lock();
        try {
            try {
                if (this.opened) {
                    throw new IllegalStateException("Storage already opened.");
                }
                logger.log(Level.FINE, "[%s] Opening Data and Index Files\u2026", this.name);
                try {
                    this.indexFile = new RandomAccessFile(new File(this.directory, String.valueOf(this.name) + ".idx"), "rws");
                    this.dataFile = new RandomAccessFile(new File(this.directory, String.valueOf(this.name) + ".dat"), "rws");
                }
                catch (FileNotFoundException fnfe1) {
                    throw new StorageException("Could not create data and/or index files!", fnfe1);
                }
                logger.log(Level.FINE, "[%s] Files opened.", this.name);
                long indexLength = this.indexFile.length();
                if (indexLength % 16L != 0L) {
                    throw new IOException("Invalid Index Length: " + indexLength);
                }
                this.emptyDirectoryEntries.clear();
                this.directoryEntries.clear();
                this.idDirectoryIndexes.clear();
                this.allocations.clear();
                logger.log(Level.FINE, "[%s] Reading " + indexLength / 16L + " existing Directory Entries\u2026", this.name);
                int directoryIndex = 0;
                while ((long)directoryIndex < indexLength / 16L) {
                    byte[] allocationBuffer = new byte[16];
                    this.indexFile.readFully(allocationBuffer);
                    Allocation allocation = Allocation.FACTORY.restore(allocationBuffer);
                    logger.log(Level.FINEST, "[%s] Read Allocation: %s", new Object[]{this.name, allocation});
                    if (allocation.getId() == 0L && allocation.getPosition() == 0 && allocation.getSize() == 0) {
                        this.emptyDirectoryEntries.set(directoryIndex);
                        this.directoryEntries.add(null);
                    } else {
                        this.directoryEntries.add(allocation);
                        this.idDirectoryIndexes.put(allocation.getId(), directoryIndex);
                        this.allocations.set(allocation.getPosition(), allocation.getPosition() + this.getBlocks(allocation.getSize()));
                    }
                    ++directoryIndex;
                }
                this.opened = true;
            }
            catch (IOException ioe1) {
                throw new StorageException("Could not open storage!", ioe1);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        logger.log(Level.INFO, "[%s] Storage opened.", this.name);
    }

    public void add(T storable) throws StorageException {
        Validation.begin().isNotNull("Storable", storable).check();
        logger.log(Level.INFO, "[%s] Adding Storable %s\u2026", new Object[]{this.name, storable});
        this.lock.writeLock().lock();
        try {
            try {
                if (!this.opened) {
                    throw new IllegalStateException("Storage not opened!");
                }
                byte[] storableBytes = storable.getBuffer();
                int storableLength = storableBytes.length;
                int blocks = this.getBlocks(storableLength);
                int position = this.findFreeRegion(blocks);
                logger.log(Level.FINEST, "[%s] Will add Storable at %d, for %d blocks.", new Object[]{this.name, position, blocks});
                logger.log(Level.FINE, "[%s] Writing Storable Data\u2026", this.name);
                this.allocations.set(position, position + blocks);
                if (this.dataFile.length() < (long)(position * this.blockSize + storableLength)) {
                    this.dataFile.setLength(position * this.blockSize + storableLength);
                }
                this.dataFile.seek(position * this.blockSize);
                this.dataFile.write(storableBytes);
                logger.log(Level.FINE, "[%s] Storable Data written.", this.name);
                int oldIndex = -1;
                Allocation allocation = new Allocation(storable.getId(), position, storableLength);
                int directoryIndex = this.emptyDirectoryEntries.nextSetBit(0);
                if (directoryIndex == -1) {
                    directoryIndex = this.directoryEntries.size();
                    this.directoryEntries.add(allocation);
                    logger.log(Level.FINEST, "[%s] Appending to Directory, Entry %d\u2026", new Object[]{this.name, directoryIndex});
                } else {
                    this.directoryEntries.set(directoryIndex, allocation);
                    this.emptyDirectoryEntries.clear(directoryIndex);
                    logger.log(Level.FINEST, "[%s] Replacing Directory Entry %d\u2026", new Object[]{this.name, directoryIndex});
                }
                if (this.idDirectoryIndexes.containsKey(storable.getId())) {
                    oldIndex = this.idDirectoryIndexes.get(storable.getId());
                    Allocation oldAllocation = this.directoryEntries.set(oldIndex, null);
                    this.emptyDirectoryEntries.set(oldIndex);
                    logger.log(Level.FINE, "[%s] Freeing Directory Index %d\u2026", new Object[]{this.name, oldIndex});
                    this.allocations.clear(oldAllocation.getPosition(), oldAllocation.getPosition() + this.getBlocks(oldAllocation.getSize()));
                }
                this.emptyDirectoryEntries.clear(directoryIndex);
                this.idDirectoryIndexes.put(storable.getId(), directoryIndex);
                this.writeAllocation(directoryIndex, allocation);
                if (oldIndex > -1) {
                    this.writeAllocation(oldIndex, null);
                }
            }
            catch (IOException ioe1) {
                throw new StorageException("Could not add Storable: " + storable + "!", ioe1);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        logger.log(Level.FINE, "[%s] Storable added.", this.name);
    }

    public int size() {
        this.lock.readLock().lock();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            int n = this.directoryEntries.size() - this.emptyDirectoryEntries.cardinality();
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public T load(long id) throws StorageException {
        Allocation allocation;
        logger.log(Level.INFO, "[%s] Loading Storable %d\u2026", new Object[]{this.name, id});
        this.lock.readLock().lock();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            Integer directoryIndex = this.idDirectoryIndexes.get(id);
            logger.log(Level.FINEST, "[%s] Directory Index: %d", new Object[]{this.name, directoryIndex});
            if (directoryIndex == null) {
                return null;
            }
            allocation = this.directoryEntries.get(directoryIndex);
            logger.log(Level.FINEST, "[%s] Allocation: %s", new Object[]{this.name, allocation});
        }
        finally {
            this.lock.readLock().unlock();
        }
        byte[] buffer = new byte[allocation.getSize()];
        this.lock.writeLock().lock();
        try {
            try {
                logger.log(Level.FINEST, "[%s] Reading %d Bytes\u2026", new Object[]{this.name, allocation.getSize()});
                this.dataFile.seek(allocation.getPosition() * this.blockSize);
                this.dataFile.readFully(buffer);
            }
            catch (IOException ioe1) {
                throw new StorageException("Could not load Storable!", ioe1);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        logger.log(Level.INFO, "[%s] Read Storable, restoring from Factory\u2026", this.name);
        return (T)((Storable)this.factory.restore(buffer));
    }

    public int getDirectorySize() {
        this.lock.readLock().lock();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            int n = this.directoryEntries.size();
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public Allocation getAllocation(int directoryIndex) {
        this.lock.readLock().lock();
        Validation.begin().isGreater("Directory Index", directoryIndex, -1L).isLess("Directory Index", directoryIndex, this.directoryEntries.size()).check();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            Allocation allocation = this.directoryEntries.get(directoryIndex);
            return allocation;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void remove(T storable) throws StorageException {
        Validation.begin().isNotNull("Storable", storable).check();
        this.remove(storable.getId());
    }

    public void remove(long id) throws StorageException {
        logger.log(Level.INFO, "[%s] Removing Storable %d\u2026", new Object[]{this.name, id});
        this.lock.writeLock().lock();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            Integer directoryIndex = this.idDirectoryIndexes.remove(id);
            logger.log(Level.FINEST, "[%s] Directory Index: %s", new Object[]{this.name, directoryIndex});
            if (directoryIndex == null) {
                return;
            }
            try {
                Allocation allocation = this.directoryEntries.set(directoryIndex, null);
                this.emptyDirectoryEntries.set(directoryIndex);
                this.allocations.clear(allocation.getPosition(), allocation.getPosition() + this.getBlocks(allocation.getSize()));
                logger.log(Level.FINE, "[%s] Clearing Directory Index %d\u2026", new Object[]{this.name, directoryIndex});
                this.indexFile.seek(directoryIndex * 16);
                this.indexFile.write(new byte[16]);
            }
            catch (IOException ioe1) {
                throw new StorageException("Could not write to index file!", ioe1);
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        logger.log(Level.FINE, "[%s] Storable removed.", this.name);
    }

    @Override
    public void close() {
        logger.log(Level.INFO, "[%s] Closing Storage\u2026", this.name);
        this.lock.writeLock().lock();
        try {
            if (!this.opened) {
                throw new IllegalStateException("Storage not opened!");
            }
            Closer.close(this.indexFile);
            Closer.close(this.dataFile);
            this.opened = false;
        }
        finally {
            this.lock.writeLock().unlock();
        }
        logger.log(Level.INFO, "[%s] Storage closed.", this.name);
    }

    public void compact() throws StorageException {
        logger.log(Level.INFO, "[%s] Compacting Storage\u2026", this.name);
        this.lock.writeLock().lock();
        try {
            try {
                if (this.opened) {
                    throw new IllegalStateException("Storage is opened!");
                }
                try {
                    this.indexFile = new RandomAccessFile(new File(this.directory, String.valueOf(this.name) + ".idx"), "rws");
                }
                catch (FileNotFoundException fnfe1) {
                    throw new StorageException("Could not create data and/or index files!", fnfe1);
                }
                long indexLength = this.indexFile.length();
                if (indexLength % 16L != 0L) {
                    throw new StorageException("Invalid Index Length: " + indexLength);
                }
                logger.log(Level.FINE, "[%s] Reading Directory\u2026", this.name);
                this.directoryEntries.clear();
                int directoryIndex = 0;
                while ((long)directoryIndex < indexLength / 16L) {
                    byte[] allocationBuffer = new byte[16];
                    this.indexFile.readFully(allocationBuffer);
                    Allocation allocation = Allocation.FACTORY.restore(allocationBuffer);
                    if (allocation.getId() == 0L && allocation.getPosition() == 0 && allocation.getSize() == 0) {
                        this.directoryEntries.add(null);
                    } else {
                        this.directoryEntries.add(allocation);
                    }
                    ++directoryIndex;
                }
                logger.log(Level.FINE, "[%s] Read %d Directory Entries.", new Object[]{this.name, this.directoryEntries.size()});
                directoryIndex = 0;
                this.indexFile.seek(0L);
                for (Allocation allocation : this.directoryEntries) {
                    if (allocation == null) continue;
                    this.writeAllocation(directoryIndex++, allocation);
                }
                logger.log(Level.FINE, "[%s] Wrote %d Directory Entries.", new Object[]{this.name, directoryIndex});
                logger.log(Level.FINE, "[%s] Truncating Directory File\u2026", this.name);
                this.indexFile.setLength(this.indexFile.getFilePointer());
            }
            catch (IOException ioe1) {
                throw new StorageException("Could not compact index!", ioe1);
            }
        }
        finally {
            Closer.close(this.indexFile);
            this.lock.writeLock().unlock();
        }
        logger.log(Level.INFO, "[%s] Storage compacted.", this.name);
    }

    private int findFreeRegion(int blocks) {
        int currentBlock = -1;
        int nextUsedBlock;
        while ((nextUsedBlock = this.allocations.nextSetBit(currentBlock + 1)) != -1 && nextUsedBlock - currentBlock - 1 < blocks) {
            currentBlock = nextUsedBlock;
        }
        return currentBlock + 1;
    }

    private int getBlocks(long size) {
        if (size == 0L) {
            return 1;
        }
        return (int)((size - 1L) / (long)this.blockSize + 1L);
    }

    private void writeAllocation(int directoryIndex, Allocation allocation) throws IOException {
        this.indexFile.seek(directoryIndex * 16);
        if (allocation == null) {
            this.indexFile.write(new byte[16]);
        } else {
            this.indexFile.write(allocation.getBuffer());
        }
    }
}

